var pewModule = require('./pew_module');
var pew = new pewModule.pewModule();
var Mew = require('./mewtocol');
var Mew7 = require('./mewtocol7');
var Modbus = require('./modbus');
const tls = require("tls");
const fs = require("fs");
const net = require("net");
const path = require('path');

var client;
var serverSocket;
var protocol;
const moduleName = "TLS Client";
const accessModes = {
    localhostOnly: 0,
    singleIP: 1,
    IPRange: 2
}
const authMethods = {
    publicKey: 0,
    clientCert: 1
}
const STATUS = {
    noErr: 0,
    connectedToTLSServer: 1,
    disconnectedFromTLSServer: 2,
    serverConnectErr: -1,
    portOccupiedErr: -2,
    serverConnectTimeout: -3,
    missingCertErr: -4,
    remoteIPDenied: -5,
    unexpectedErr: -99
}

//Catch exceptions
process.on('uncaughtException', function (err) {
    pew.sysLogMessage(moduleName, "Uncaught exception: " + err.stack);
})

//Read Configurations
var interfaceConfig = pew.getConfFileSync(pew.Constants.requests.READ_INTERFACE);
var portConfig = pew.getConfFileSync(pew.Constants.requests.READ_PORTS);
var tlsConfig = pew.getConfFileSync(pew.Constants.requests.READ_TLS);

//If any configuration could not be loaded -> log the message and stop execution
if (interfaceConfig.err) {
    pew.sysLogMessage(moduleName, interfaceConfig.err_msg);
    return;
}

if (portConfig.err) {
    pew.sysLogMessage(moduleName, portConfig.err_msg);
    return;
}

if (tlsConfig.err) {
    pew.sysLogMessage(moduleName, tlsConfig.err_msg);
    return;
}

//Check configurations
var usePort, useProtocol, serviceID;
serviceID = interfaceConfig.data.interface_plc;
let iface = interfaceConfig.data.interface.filter(val => {
    return val.service == serviceID;
})

if (iface.length <= 0 || !iface[0].enabled) {
    return pew.sysLogMessage(moduleName, pew.Constants.COM_NOT_ENABLED);
}
useProtocol = iface[0].protocol;
//Set the protocol module
switch (useProtocol) {
    case pew.Constants.PROTOCOLS.mewtocol:
        protocol = new Mew.Mewtocol();
        break;

    case pew.Constants.PROTOCOLS.mewtocol7:
        protocol = new Mew7.Mewtocol7();
        break;

    case pew.Constants.PROTOCOLS.modbus:
        protocol = new Modbus.Modbus();
        break;

    default:
        protocol = new Mew.Mewtocol();
}

var thread = portConfig.data.thread.filter(val => {
    return val.service === serviceID;
});

if (thread.length <= 0) {
    return pew.sysLogMessage(moduleName, pew.Constants.TCP_PORT_DISABLED);
}
usePort = thread[0].port;

if (pew.isPortOccupied(tlsConfig.data.nontls_server_port)) {
    setPLCStatus([STATUS.noErr, STATUS.portOccupiedErr]);
    return pew.sysLogMessage(moduleName, `${pew.Constants.TCP_PORT_OCCUPIED} -- ${tlsConfig.data.nontls_server_port}`);
}

if (tlsConfig.data.tlsclient_certauth === authMethods.clientCert && (tlsConfig.data.tlsclientcert === pew.Constants.EMPTY_STRING || tlsConfig.data.tlsclientkey === pew.Constants.EMPTY_STRING)) {
    setPLCStatus([STATUS.noErr, STATUS.missingCertErr]);
    return pew.sysLogMessage(moduleName, pew.Constants.CERT_FILE_MISSING);
}
//Configurations all ok, then continue

//Set minimum TLS Version
tls.DEFAULT_MIN_VERSION = tlsConfig.data.tls_min_version_client;

var listeningServer = tlsConfig.data.nontls_accessability == accessModes.localhostOnly ? pew.Constants.LOCAL_HOST : pew.Constants.ALL_IP;
/***************************************************************************************************************************/

/********************************************************************************/
//Server - no encryption
/********************************************************************************/
var server = net.createServer(function (socket) {
    serverSocket = socket;
    serverSocket.on("data", function (data) {
        let allowSendData = false;
        //Check if IP accessibility is limited
        switch (tlsConfig.data.nontls_accessability) {
            case accessModes.localhostOnly:
                allowSendData = true;
                break;
            case accessModes.singleIP:
                if (serverSocket.remoteAddress === tlsConfig.data.nontls_access_ip1)
                    allowSendData = true;
                break;

            case accessModes.IPRange:
                let remoteIP = pew.ip2int(serverSocket.remoteAddress);
                let ip1 = pew.ip2int(tlsConfig.data.nontls_access_ip1);
                let ip2 = pew.ip2int(tlsConfig.data.nontls_access_ip2);
                if (remoteIP >= ip1 && remoteIP <= ip2)
                    allowSendData = true;
                break;
        }

        if (allowSendData) {
            //Check if client is connected
            if (client.getSession()) {
                client.write(data);
            }
            else {
                connectToTLSServer().then(() => {
                    client.write(data);
                });
            }
        }
        else {
            setPLCStatus([STATUS.noErr, STATUS.remoteIPDenied]);
            pew.sysLogMessage(moduleName, `Remote IP (${serverSocket.remoteAddress}) not allowed`);
        }
    })
});

server.listen(tlsConfig.data.nontls_server_port, listeningServer);
/********************************************************************************/

/********************************************************************************/
//Client
/********************************************************************************/
const options = {
    host: tlsConfig.data.tls_server_host,
    port: tlsConfig.data.tls_destination_port,
                                    
    // Necessary only if using the client certificate authentication
    key: tlsConfig.data.tlsclient_certauth === authMethods.clientCert ? fs.readFileSync(
        path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsclientkey)) : "",
    cert: tlsConfig.data.tlsclient_certauth === authMethods.clientCert ? fs.readFileSync(
        path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsclientcert)) : "",

    // Necessary only if the server uses the self-signed certificate
    ca: tlsConfig.data.tlsclient_certauth === authMethods.clientCert && tlsConfig.data.tlsclientca !== pew.Constants.EMPTY_STRING ? fs.readFileSync(
        path.join(pew.Constants.configSubfolders.tls, tlsConfig.data.tlsclientca)) : "",
    rejectUnauthorized: tlsConfig.data.ca_verification_tlsclient ? true : false,
};
//Connect to TLS Server
connectToTLSServer().catch(err => {
    //Just in case the server is not reachable...
    pew.sysLogMessage(moduleName, err);
});

//Start plc polling
plcPolling();

/*******************************************************************************/
function connectToTLSServer() {
    return new Promise((res, rej) => {
        try {
            client = tls
                .connect(options, () => {
                    //Connected...
                    setPLCStatus([STATUS.noErr, STATUS.connectedToTLSServer]);
                    res();
                })
                .on("data", (data) => {
                    //Send back data to the client which is not encrypted
                    if (serverSocket)
                        serverSocket.write(data);
                })
                .on("timeout", () => {
                    setPLCStatus([STATUS.noErr, STATUS.serverConnectTimeout]);
                    pew.sysLogMessage(moduleName, "Connect to TLS server timeout");
                })
                .on("end", () => {
                    pew.sysLogMessage(moduleName, "TLS server end connection");
                    setPLCStatus([STATUS.noErr, STATUS.disconnectedFromTLSServer]);
                })
                .on("error", (error) => {
                    setPLCStatus([STATUS.noErr, STATUS.serverConnectErr]);
                    //pew.sysLogMessage(moduleName, `TLS connection error: ${error}`);
                    disconnectFromTLSServer();
                    rej(error);
                });
        }
        catch (ex) {
            setPLCStatus([STATUS.noErr, STATUS.unexpectedErr]);
            //pew.sysLogMessage(moduleName, ex.message);
            rej(ex);
        }

    })

}

function disconnectFromTLSServer() {
    try {
        client.end();
        client.destroy();
        setPLCStatus([STATUS.noErr, STATUS.disconnectedFromTLSServer]);
    }
    catch (ex) {
        pew.sysLogMessage(moduleName, ex.message);
    }
}

function plcPolling() {
    let readItem = {
        area: "DT",
        type: "INT",
        count: 1,
        start: tlsConfig.data.plc_control_dt
    }

    const control = {
        reconnect: 1,
        disconnect: 2
    }

    protocol.StartAddr = readItem.start;
    pew.readMultipleRegisters(protocol, usePort, readItem).then(data => {       
        if (!data.err) {
            switch (data.int[0]) {
                case control.reconnect:
                    disconnectFromTLSServer();
                    connectToTLSServer().catch(err => {                        
                        //Just in case the server is not reachable...
                        pew.sysLogMessage(moduleName, err);
                        setPLCStatus([STATUS.noErr, STATUS.serverConnectErr]);
                    });
                    break;

                case control.disconnect:
                    disconnectFromTLSServer();
                    break;
            }
        }
        else {
            pew.sysLogMessage(moduleName, data.err_msg);
        }
    }).catch(err => {
        pew.sysLogMessage(err);
    }).finally(() => {
        setTimeout(plcPolling, tlsConfig.data.plc_poll);
    })
}

function setPLCStatus(statusValues) {
    let setStatus = {
        area: "DT",
        type: "INT",
        count: 2,
        start: tlsConfig.data.plc_control_dt
    }
    protocol.StartAddr = setStatus.start;
    setStatus.value = statusValues;
    //Service 5 is Ethernet. In case of ethernet interface, don't use the station number,
    //instead use EE
    if (thread[0].service !== 5) {
        protocol.Station = interfaceConfig.data.address_plc;
    }  
    pew.writeMultipleRegisters(protocol, usePort, setStatus);
}